// routes/contactRoutes.ts
import express from "express";
import { contactController } from "./contact.controller";
import {
  isAuthenticated,
  hasPermission,
  isAdmin,
} from "../../middlewares/auth";

const router = express.Router();

// Public routes
router.get("/", contactController.getContactPageData);
router.post("/message", contactController.createContactMessage);

// Protected routes (Admin only)
router.get(
  "/message",
  isAuthenticated,
  isAdmin,
  contactController.getAllMessages,
);
router.get(
  "/message/stats",
  isAuthenticated,
  isAdmin,
  contactController.getMessageStats,
);
router.get(
  "/message/follow-up",
  isAuthenticated,
  isAdmin,
  contactController.getFollowUpMessages,
);
router.get(
  "/message/email/:email",
  isAuthenticated,
  isAdmin,
  contactController.getMessagesByEmail,
);
router.get(
  "/message/:id",
  isAuthenticated,
  isAdmin,
  contactController.getMessageById,
);

// Message management (Admin only)
router.put(
  "/message/:id",
  isAuthenticated,
  isAdmin,
  contactController.updateMessage,
);
router.delete(
  "/message/:id",
  isAuthenticated,
  isAdmin,
  contactController.deleteMessage,
);
router.delete(
  "/message/delete-multiple",
  isAuthenticated,
  isAdmin,
  contactController.deleteMultipleMessages,
);
router.put(
  "/message/bulk-update",
  isAuthenticated,
  isAdmin,
  contactController.bulkUpdateMessageStatus,
);
router.put(
  "/message/:id/mark-spam",
  isAuthenticated,
  isAdmin,
  contactController.markMessageAsSpam,
);

// Reply to message (Admin only)
router.post(
  "/message/reply",
  isAuthenticated,
  isAdmin,
  contactController.replyToMessage,
);

// Contact settings (Admin only)
router.put(
  "/",
  isAuthenticated,
  isAdmin,
  contactController.updateContactSettings,
);

// Test email (Admin only)
router.post(
  "/test-email",
  isAuthenticated,
  isAdmin,
  contactController.testEmailSending,
);

export default router;
