import express from "express";
import { productGalleryController } from "./product.gallery.controller";
import {
  isAuthenticated,
  hasPermission,
  canModifyResource,
} from "../../middlewares/auth";

const router = express.Router();

// Public routes
router.get("/product/:productId", productGalleryController.getProductGallery);
router.get(
  "/product/:productId/featured",
  productGalleryController.getFeaturedGallery,
);
router.get(
  "/product/:productId/primary",
  productGalleryController.getPrimaryImage,
);
router.get(
  "/product/:productId/stats",
  productGalleryController.getGalleryStats,
);
router.get(
  "/product/:productId/type/:type",
  productGalleryController.getGalleryByType,
);
router.get("/:id", productGalleryController.getGalleryItemById);

// Protected routes - require authentication
router.post(
  "/upload",
  isAuthenticated,
  productGalleryController.uploadGalleryItem,
);

router.post(
  "/upload/direct",
  isAuthenticated,
  productGalleryController.directUpload,
);

router.post(
  "/upload/multiple",
  isAuthenticated,
  productGalleryController.uploadMultipleItems,
);

router.put(
  "/:id",
  isAuthenticated,
  canModifyResource(
    "gallery",
    "gallery:update",
    "gallery:delete",
    "product_creator_id",
  ),
  productGalleryController.updateGalleryItem,
);

router.delete(
  "/:id",
  isAuthenticated,
  canModifyResource(
    "gallery",
    "gallery:update",
    "gallery:delete",
    "product_creator_id",
  ),
  productGalleryController.deleteGalleryItem,
);

// Bulk operations
router.post(
  "/bulk/delete",
  isAuthenticated,
  productGalleryController.bulkDeleteGalleryItems,
);

// Product-specific operations
router.post(
  "/product/:productId/reorder",
  isAuthenticated,
  productGalleryController.reorderGalleryItems,
);

router.post(
  "/product/:productId/featured",
  isAuthenticated,
  productGalleryController.setFeaturedItems,
);

router.post(
  "/product/:productId/primary",
  isAuthenticated,
  productGalleryController.setPrimaryImage,
);

router.post(
  "/product/:productId/update-logo",
  isAuthenticated,
  productGalleryController.updateProductLogoFromPrimary,
);

export default router;
