// routes/productAlternativeRoutes.ts
import express from "express";
import {
  isAuthenticated,
  hasPermission,
  hasAnyPermission,
  canModifyResource,
} from "../../middlewares/auth";
import { productAlternativeController } from "./product.alternative.controller";

const router = express.Router();

// Public routes
router.get("/", productAlternativeController.getAllAlternatives);
router.get("/:id", productAlternativeController.getAlternativeById);
router.get(
  "/product/:productId/stats",
  productAlternativeController.getAlternativeStats,
);
router.get(
  "/product/:productId/featured",
  productAlternativeController.getFeaturedAlternatives,
);
router.get(
  "/product/:productId/price/:priceComparison",
  productAlternativeController.getAlternativesByPrice,
);
router.get(
  "/compare/:productId/:alternativeProductId",
  productAlternativeController.getProductComparison,
);

// Protected routes - require authentication
router.post(
  "/",
  isAuthenticated,
  hasAnyPermission(["product:create", "product:manage_alternatives"]),
  productAlternativeController.createAlternative,
);

router.post(
  "/manual",
  isAuthenticated,
  hasAnyPermission(["product:create", "product:manage_alternatives"]),
  productAlternativeController.addManualAlternative,
);

router.put(
  "/:id",
  isAuthenticated,
  productAlternativeController.updateAlternative,
);

router.delete(
  "/:id",
  isAuthenticated,
  productAlternativeController.deleteAlternative,
);

// Alternative interactions
router.post(
  "/:id/vote",
  isAuthenticated,
  productAlternativeController.voteOnAlternative,
);

router.post(
  "/:id/confirm",
  isAuthenticated,
  productAlternativeController.confirmAlternative,
);

router.post(
  "/:id/reject",
  isAuthenticated,
  productAlternativeController.rejectAlternative,
);

// Product-specific routes
router.get(
  "/product/:productId/user-view",
  isAuthenticated,
  productAlternativeController.getAlternativesForUser,
);

router.post(
  "/product/:productId/auto-suggest",
  isAuthenticated,
  hasPermission("product:manage_alternatives"),
  productAlternativeController.autoSuggestAlternatives,
);

router.put(
  "/product/:productId/reorder",
  isAuthenticated,
  hasPermission("product:manage_alternatives"),
  productAlternativeController.reorderAlternatives,
);

// Admin routes
router.put(
  "/bulk-update",
  isAuthenticated,
  hasPermission("product:bulk_update"),
  productAlternativeController.bulkUpdateAlternatives,
);

// Permission check
router.get(
  "/:id/check-permission",
  isAuthenticated,
  productAlternativeController.checkAlternativePermission,
);

export default router;
