// routes/productTeamRoutes.ts
import express from "express";
import { productTeamController } from "./product.team.controller";
import {
  isAuthenticated,
  hasPermission,
  hasAnyPermission,
} from "../../middlewares/auth";

const router = express.Router();

// Public routes
router.get("/", productTeamController.getAllTeamMembers);
router.get("/:id", productTeamController.getTeamMemberById);
router.get(
  "/product/:productId/summary",
  productTeamController.getProductTeamSummary,
);
router.get(
  "/product/:productId/enriched",
  productTeamController.getEnrichedTeamData,
);
router.get(
  "/product/:productId/analytics",
  productTeamController.getTeamAnalytics,
);
router.get(
  "/product/:productId/role/:role",
  productTeamController.getTeamByRole,
);
router.get("/search", productTeamController.searchTeamMembers);

// Protected routes - require authentication
router.post(
  "/",
  isAuthenticated,
  hasAnyPermission(["product:manage_team", "product:create"]),
  productTeamController.addTeamMember,
);

router.post(
  "/product/:productId/bulk-add",
  isAuthenticated,
  hasPermission("product:manage_team"),
  productTeamController.bulkAddTeamMembers,
);

router.put("/:id", isAuthenticated, productTeamController.updateTeamMember);

router.delete("/:id", isAuthenticated, productTeamController.removeTeamMember);

router.delete(
  "/:id/leave",
  isAuthenticated,
  productTeamController.leaveProductTeam,
);

// User-specific routes
router.get(
  "/user/portfolio",
  isAuthenticated,
  productTeamController.getUserProductPortfolio,
);

router.get(
  "/user/portfolio/:userId",
  isAuthenticated,
  productTeamController.getUserProductPortfolio,
);

router.get(
  "/user/stats",
  isAuthenticated,
  productTeamController.getTeamMemberStats,
);

router.get(
  "/user/stats/:userId",
  isAuthenticated,
  productTeamController.getTeamMemberStats,
);

router.get(
  "/user/teams",
  isAuthenticated,
  productTeamController.getMyProductTeams,
);

// Product team membership check
router.get(
  "/product/:productId/check-membership",
  isAuthenticated,
  productTeamController.checkUserInProductTeam,
);

// Permission check
router.get(
  "/:id/check-permission",
  isAuthenticated,
  productTeamController.checkTeamMemberPermission,
);

// Reorder team members
router.put(
  "/product/:productId/reorder",
  isAuthenticated,
  hasPermission("product:manage_team"),
  productTeamController.reorderTeamMembers,
);

// Update last active
router.put(
  "/:id/last-active",
  isAuthenticated,
  productTeamController.updateLastActive,
);

// Admin operations
router.put(
  "/:id/verify",
  isAuthenticated,
  hasPermission("product:manage_team"),
  productTeamController.verifyTeamMember,
);

router.put(
  "/:id/feature",
  isAuthenticated,
  hasPermission("product:manage_team"),
  productTeamController.featureTeamMember,
);

export default router;
