import express from "express";
import { productReviewController } from "./product.review.controller";
import {
  isAuthenticated,
  hasPermission,
  hasAnyPermission,
  canModifyResource,
} from "../../middlewares/auth";

const router = express.Router();

// Public routes
router.get("/", productReviewController.getAllReviews);
router.get("/recent", productReviewController.getRecentReviews);
router.get("/product/:productId", productReviewController.getAllReviews);
router.get(
  "/product/:productId/stats",
  productReviewController.getProductReviewStats,
);
router.get(
  "/product/:productId/distribution",
  productReviewController.getProductRatingDistribution,
);
router.get(
  "/product/:productId/top-helpful",
  productReviewController.getTopHelpfulReviews,
);
router.get(
  "/product/:productId/verified",
  productReviewController.getVerifiedPurchaseReviews,
);
router.get(
  "/product/:productId/rating/:rating",
  productReviewController.getReviewsByRating,
);
router.get("/:id", productReviewController.getReviewById);

// Protected routes - require authentication
router.post("/", isAuthenticated, productReviewController.createReview);

router.get(
  "/user/reviews",
  isAuthenticated,
  productReviewController.getUserReviews,
);

router.get(
  "/product/:productId/user",
  isAuthenticated,
  productReviewController.getUserProductReview,
);

router.get(
  "/product/:productId/can-review",
  isAuthenticated,
  productReviewController.canUserReview,
);

router.put(
  "/:id",
  isAuthenticated,
  canModifyResource("review", "review:update", "review:delete", "creator_id"),
  productReviewController.updateReview,
);

router.delete(
  "/:id",
  isAuthenticated,
  canModifyResource("review", "review:update", "review:delete", "creator_id"),
  productReviewController.deleteReview,
);

// Review interactions
router.post(
  "/:id/helpful",
  isAuthenticated,
  productReviewController.markReviewHelpful,
);

// Moderator/Admin routes
router.get(
  "/moderation/pending",
  isAuthenticated,
  hasAnyPermission(["review:moderate", "review:approve"]),
  productReviewController.getPendingReviews,
);

router.put(
  "/:id/moderate",
  isAuthenticated,
  hasAnyPermission(["review:moderate", "review:approve"]),
  productReviewController.moderateReview,
);

router.post(
  "/bulk/update",
  isAuthenticated,
  hasPermission("review:bulk_update"),
  productReviewController.bulkUpdateReviews,
);

// Admin analytics
router.get(
  "/analytics/summary",
  isAuthenticated,
  hasPermission("review:analytics"),
  productReviewController.getReviewSummary,
);

router.get(
  "/analytics/detailed",
  isAuthenticated,
  hasPermission("review:analytics"),
  productReviewController.getReviewAnalytics,
);

export default router;
