import express from "express";
import { productCommentController } from "./product.comment.controller";
import {
  isAuthenticated,
  isAdmin,
  hasPermission,
  canModifyResource,
} from "../../middlewares/auth";

const router = express.Router();

// Public routes
router.get("/product/:productId", productCommentController.getProductComments);
router.get(
  "/product/:productId/stats",
  productCommentController.getProductCommentStats,
);
router.get("/recent", productCommentController.getRecentComments);
router.get("/thread/:id", productCommentController.getCommentThread);
router.get("/:id", productCommentController.getCommentById);

// Protected routes - require authentication
router.post("/", isAuthenticated, productCommentController.createComment);

router.put(
  "/:id",
  isAuthenticated,
  canModifyResource(
    "comment",
    "comment:update",
    "comment:delete",
    "creator_id",
  ),
  productCommentController.updateComment,
);

router.delete(
  "/:id",
  isAuthenticated,
  canModifyResource(
    "comment",
    "comment:update",
    "comment:delete",
    "creator_id",
  ),
  productCommentController.deleteComment,
);

// Comment interactions
router.post(
  "/:id/upvote",
  isAuthenticated,
  productCommentController.upvoteComment,
);

router.post(
  "/:id/downvote",
  isAuthenticated,
  productCommentController.downvoteComment,
);

router.post(
  "/:id/report",
  isAuthenticated,
  productCommentController.reportComment,
);

// User comments
router.get(
  "/user/comments",
  isAuthenticated,
  productCommentController.getUserComments,
);

router.get(
  "/user/mentions",
  isAuthenticated,
  productCommentController.getCommentsWithMentions,
);

router.get(
  "/product/:productId/user",
  isAuthenticated,
  productCommentController.getUserProductComment,
);

// Admin and moderator routes
router.put(
  "/:id/pin",
  isAuthenticated,
  hasPermission("comment:pin"),
  productCommentController.pinComment,
);

router.post(
  "/bulk/delete",
  isAuthenticated,
  hasPermission("comment:bulk_delete"),
  productCommentController.bulkDeleteComments,
);

router.put(
  "/:id/restore",
  isAuthenticated,
  hasPermission("comment:restore"),
  productCommentController.restoreComment,
);

export default router;
