import express from "express";
import { productUpdateController } from "./product.update.controller";
import {
  isAuthenticated,
  hasPermission,
  hasAnyPermission,
} from "../../middlewares/auth";

const router = express.Router();

// Public routes
router.get("/", productUpdateController.getAllUpdates);
router.get("/recent", productUpdateController.getRecentUpdates);
router.get("/search", productUpdateController.searchUpdates);
router.get("/product/:productId", productUpdateController.getProductUpdates);
router.get(
  "/product/:productId/latest",
  productUpdateController.getLatestUpdate,
);
router.get("/product/:productId/stats", productUpdateController.getUpdateStats);
router.get(
  "/product/:productId/history",
  productUpdateController.getVersionHistory,
);
router.get(
  "/product/:productId/changelog",
  productUpdateController.getChangelog,
);
router.get(
  "/product/:productId/timeline",
  productUpdateController.getUpdateTimeline,
);
router.get(
  "/product/:productId/type/:type",
  productUpdateController.getUpdatesByType,
);
router.get(
  "/product/:productId/version/:version",
  productUpdateController.getUpdateByVersion,
);
router.get("/:id", productUpdateController.getUpdateById);
router.get("/slug/:slug", productUpdateController.getUpdateBySlug);
router.get(
  "/compare/:updateId1/:updateId2",
  productUpdateController.compareUpdates,
);

// Protected routes - require authentication
router.post("/", isAuthenticated, productUpdateController.createUpdate);

router.get(
  "/user/updates",
  isAuthenticated,
  productUpdateController.getUserUpdates,
);

router.get(
  "/product/:productId/new-updates",
  isAuthenticated,
  productUpdateController.getNewUpdatesSince,
);

router.put("/:id", isAuthenticated, productUpdateController.updateUpdate);

router.delete("/:id", isAuthenticated, productUpdateController.deleteUpdate);

// Update interactions
router.post(
  "/:id/upvote",
  isAuthenticated,
  productUpdateController.upvoteUpdate,
);

router.post(
  "/:id/set-latest",
  isAuthenticated,
  productUpdateController.setAsLatest,
);

// Admin/Moderator routes
router.post(
  "/bulk/publish",
  isAuthenticated,
  hasAnyPermission(["update:publish", "update:moderate"]),
  productUpdateController.bulkPublishUpdates,
);

export default router;
